#region References

using System;
using System.Data;
using System.Collections;
using System.Text;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.ExceptionManagement;

#endregion

namespace gov.va.med.vbecs.BOL
{
	#region Header
	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>John Vrooland</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>3/12/2003</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///
	/// <summary>
	/// Inactivates a blood unit
	/// </summary>
	/// 
	#endregion

	public class BloodUnitInactivate : BaseBusinessObject
	{
		#region Static Methods

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/3/2005</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="3740"> 
		///		<ExpectedInput>A blood unit in final status</ExpectedInput>
		///		<ExpectedOutput>Common.StrRes.SysErrMsg.UC070.UnitIsInFinalStatus().ResString</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="3741"> 
		///		<ExpectedInput>A blood unit that has a history of patient assignments and user's security privileges are below VBECS security level 4</ExpectedInput>
		///		<ExpectedOutput>string.Concat("Unit has previous history of patient assignment. You cannot modify this unit. \n", Common.StrRes.SysErrMsg.Common.InsufficientSecurityPrivileges("inactivate a blood unit").ResString)</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="3742"> 
		///		<ExpectedInput>A blood unit that's currently issued to a patient </ExpectedInput>
		///		<ExpectedOutput>Common.StrRes.SysErrMsg.UC070.CannotInactivateUnit().ResString</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="3743"> 
		///		<ExpectedInput>A blood unit that has active tests</ExpectedInput>
		///		<ExpectedOutput>Common.StrRes.SysErrMsg.UC070.UnitHasActiveTests().ResString</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3744"> 
		///		<ExpectedInput>A blood unit and an empty comment</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Determines whether a blood unit is eligible for inactivation
		/// Implements BR_70.06, BR_70.07, BR_70.11, BR_70.21
		/// </summary>
		/// <param name="bloodUnit">The blood unit to be inactivated</param>
		public static string IsUnitEligibleForInactivation(BloodUnit bloodUnit) 
		{
			StringBuilder message = new StringBuilder(string.Empty);
			
			// ***************************************************************************************************************
			// If any of the following conditions are met, the unit cannot be inactivated:
			// ***************************************************************************************************************
			//
			// BR_70.06
			if (bloodUnit.BloodUnitStatus.IsFinal())
			{
				message.Append("\n");
				message.Append(Common.StrRes.SysErrMsg.UC070.UnitIsInFinalStatus().ResString);
			}
			// BR_70.07
			DataTable previousOrders = BOL.OrderedUnit.GetHistoricalOrderedUnits(bloodUnit.BloodUnitGuid, Common.LogonUser.LogonUserDivisionCode);
			if (!VbecsUser.LoggedOnUser.IsAuthorized(Common.UserRoleType.TraditionalSupervisor) && previousOrders.Rows.Count > 0)
			{
				message.Append("\n");
				message.Append("Unit has previous history of patient assignment. You cannot modify this unit.");
			}
			// BR_70.11
			if (bloodUnit.PatientAssignments.Rows.Count > 0)
			{
				message.Append("\n");
				message.Append(Common.StrRes.SysErrMsg.UC070.CannotInactivateUnit().ResString);
			}
			// BR_70.21
			ArrayList alAboRh = BOL.BloodUnitTest.GetAboRHConfirmationTests(bloodUnit.BloodUnitGuid, Common.LogonUser.LogonUserDivisionCode);
			ArrayList alAT = BOL.BloodUnitTest.GetAntigenTypingTests(bloodUnit.BloodUnitGuid, Common.LogonUser.LogonUserDivisionCode);
			if ((alAboRh.Count > 0) || (alAT.Count > 0))
			{
				message.Append("\n");
				message.Append(Common.StrRes.SysErrMsg.UC070.UnitHasActiveTests().ResString);
			}
			//
			return message.ToString();
		}

		///<Developers>
		///	<Developer>Martin Tsotchev</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/16/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="6091"> 
		///		<ExpectedInput>Valid BloodUnit</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6092"> 
		///		<ExpectedInput>Invalid BloodUnit</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Inactivates a blood unit
		/// Implements BR_70.09, BR_70.17, BR_70.18
		/// </summary>
		/// <param name="bloodUnit">The blood unit to be inactivated</param>
		/// <param name="inactivateCommentText">A comment required to inactivate a unit</param>
		/// <param name="lastUpdateFunctionId">UC or function performing this call</param>
		/// <param name="dtBloodUnitAntigens">Antigens of bloodUnit</param>
		/// <param name="inactivationWorkload">Returns the calculated workload process id</param>
		public static bool InactivateUnit
		(
			BOL.BloodUnit bloodUnit, 
			string inactivateCommentText, 
			Common.UpdateFunction lastUpdateFunctionId, 
			ArrayList dtBloodUnitAntigens,
			ref Common.WorkloadProcessID inactivationWorkload
		) 
		{
			
			// Generate workload data for log in and log out of a unit
			Guid [] bloodUnitGuids = new Guid[1];
			bloodUnitGuids[0] = bloodUnit.BloodUnitGuid;
			
			//checking if the unit was modified and selecting the appropriate workload as defined in TT_15.01
			DataTable dtBloodUnitModifications = BOL.BloodUnitModification.GetBloodUnitModificationsForChildUnit(bloodUnit.BloodUnitGuid);
			if (dtBloodUnitModifications.Rows.Count == 0)
			{
				inactivationWorkload = Common.WorkloadProcessID.UnitLogin;
			}
			else
			{
				if (dtBloodUnitModifications.Rows[0][Common.VbecsTables.BloodUnitModification.ProductModificationCode].ToString().Trim() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Pooled).ToString())
				{
					inactivationWorkload = Common.WorkloadProcessID.PoolUnit;
				}
				else if  (dtBloodUnitModifications.Rows[0][Common.VbecsTables.BloodUnitModification.ProductModificationCode].ToString().Trim() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.PoolThawCryo).ToString())
				{
					inactivationWorkload = Common.WorkloadProcessID.ThawPoolCryo;
				}
				else
				{
					inactivationWorkload = Common.WorkloadProcessID.UnitLogin;
				}
			}
			DataTable dtWorkloadEvent = BOL.WorkloadEvent.GenerateWorkloadData(inactivationWorkload, bloodUnitGuids, Common.WorkloadTransactionType.Unit, true);
			dtWorkloadEvent = Common.Utility.AppendLastUpdateInformation(dtWorkloadEvent, lastUpdateFunctionId);
			// BR_70.09, BR_70.17, BR_70.18
			//

			System.Data.DataTable dtBloodUnit = DAL.BloodUnit.GetEmptyBloodUnitTableSchema(true);
			System.Data.DataRow drUnit = bloodUnit.LoadDataRowFromThis(dtBloodUnit.NewRow());
			drUnit[Common.VbecsTables.BloodUnit.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
			dtBloodUnit.Rows.Add(drUnit);
			dtBloodUnit = Common.Utility.AppendLastUpdateInformation(dtBloodUnit, lastUpdateFunctionId);


			System.Data.DataTable dtBloodUnitMedia = DAL.BloodUnitMedia.GetEmptyBloodUnitMediaTableSchema(true);
			System.Data.DataRow drUnitMedia = bloodUnit.BloodUnitMedia.LoadDataRowFromThis(dtBloodUnitMedia.NewRow());
			drUnitMedia[Common.VbecsTables.BloodUnitMedia.Comments] = inactivateCommentText;
			dtBloodUnitMedia.Rows.Add(drUnitMedia);
			dtBloodUnitMedia = Common.Utility.AppendLastUpdateInformation(dtBloodUnitMedia, lastUpdateFunctionId);

			DataTable dtBloodUnitAntigen = new DataTable();
			dtBloodUnitAntigen.Columns.Add(Common.VbecsTables.BloodUnitAntigen.BloodUnitGuid, typeof(System.Guid));
			dtBloodUnitAntigen.Columns.Add(Common.VbecsTables.BloodUnitAntigen.Comments, typeof(string));
			dtBloodUnitAntigen.Columns.Add(Common.VbecsTables.BloodUnitAntigen.DivisionCode, typeof(string));
			dtBloodUnitAntigen.Columns.Add(Common.VbecsTables.BloodUnitAntigen.LastUpdateDate, typeof(System.DateTime));
			dtBloodUnitAntigen.Columns.Add(Common.VbecsTables.BloodUnitAntigen.LastUpdateUser, typeof(string));
			dtBloodUnitAntigen.Columns.Add(Common.VbecsTables.BloodUnitAntigen.LastUpdateFunctionId, typeof(int));
			dtBloodUnitAntigen.Columns.Add(Common.VbecsTables.BloodUnitAntigen.RowVersion, typeof(byte[]));
			
			foreach(object row in dtBloodUnitAntigens)
			{
				DataRow dtAntigenRow = dtBloodUnitAntigen.NewRow();
				dtAntigenRow[Common.VbecsTables.BloodUnitAntigen.BloodUnitGuid] = bloodUnit.BloodUnitGuid;
				dtAntigenRow[Common.VbecsTables.BloodUnitAntigen.Comments] = inactivateCommentText;
				dtAntigenRow[Common.VbecsTables.BloodUnitAntigen.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
				dtAntigenRow[Common.VbecsTables.BloodUnitAntigen.LastUpdateDate] = VBECSDateTime.GetDivisionCurrentDateTime();
				dtAntigenRow[Common.VbecsTables.BloodUnitAntigen.LastUpdateUser] = Common.LogonUser.LogonUserName;
				dtAntigenRow[Common.VbecsTables.BloodUnitAntigen.LastUpdateFunctionId] = (int)lastUpdateFunctionId;
				dtAntigenRow[Common.VbecsTables.BloodUnitAntigen.RowVersion] = ((BloodUnitAntigen)row).RowVersion;				
				dtBloodUnitAntigen.Rows.Add(dtAntigenRow);
			}
			//
			// Inactivate Blood Unit Shipment record for Pools and Splits
			if( DAL.BloodUnitModification.WasBloodUnitPooledOrSplit( bloodUnit.BloodUnitGuid ) )
			{  
				// Need BloodUnitShipmentGuid and RowVersion
				DataRow bloodUnitShipmentRow = DAL.BloodUnitModification.GetMostRecentBloodUnitShipmentRecordForBloodUnit( bloodUnit.BloodUnitGuid );
				//
				DataTable dtBloodUnitShipment = new DataTable();
				dtBloodUnitShipment.Columns.Add(Common.VbecsTables.BloodUnitShipment.BloodUnitShipmentGuid, typeof(System.Guid));
				dtBloodUnitShipment.Columns.Add(Common.VbecsTables.BloodUnitShipment.LastUpdateDate, typeof(System.DateTime));
				dtBloodUnitShipment.Columns.Add(Common.VbecsTables.BloodUnitShipment.LastUpdateUser, typeof(string));
				dtBloodUnitShipment.Columns.Add(Common.VbecsTables.BloodUnitShipment.LastUpdateFunctionId, typeof(int));
				dtBloodUnitShipment.Columns.Add(Common.VbecsTables.BloodUnitShipment.RowVersion, typeof(byte[]));
			
				DataRow drBloodUnitShipment = dtBloodUnitShipment.NewRow();
				drBloodUnitShipment[Common.VbecsTables.BloodUnitShipment.BloodUnitShipmentGuid] = bloodUnitShipmentRow[Common.VbecsTables.BloodUnitShipment.BloodUnitShipmentGuid];
				drBloodUnitShipment[Common.VbecsTables.BloodUnitShipment.LastUpdateDate] = VBECSDateTime.GetDivisionCurrentDateTime();
				drBloodUnitShipment[Common.VbecsTables.BloodUnitShipment.LastUpdateUser] = Common.LogonUser.LogonUserName;
				drBloodUnitShipment[Common.VbecsTables.BloodUnitShipment.LastUpdateFunctionId] = (int)lastUpdateFunctionId;
				drBloodUnitShipment[Common.VbecsTables.BloodUnitShipment.RowVersion] = bloodUnitShipmentRow[Common.VbecsTables.BloodUnitShipment.RowVersion];
				dtBloodUnitShipment.Rows.Add(drBloodUnitShipment);
				// 
				return( DAL.BloodUnitInactivate.InactivateUnit( dtBloodUnit, dtBloodUnitMedia, dtBloodUnitAntigen, dtBloodUnitShipment, dtWorkloadEvent ) );
			}
			else
			{
				return( DAL.BloodUnitInactivate.InactivateUnit( dtBloodUnit, dtBloodUnitMedia, dtBloodUnitAntigen, dtWorkloadEvent ) );
			}
		}

		#endregion

		#region Implementation of IBaseBusinessObject

		///<Developers>
		///	<Developer>Martin Tsotchev</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1655"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1656"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// LoadDataRowFromThis
		/// </summary>
		/// <param name="dtRow">dtRow</param>
		/// <returns></returns>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			return dtRow;
		}

		/// <summary>
		/// LoadFromDataRow
		/// </summary>
		/// <param name="dtRow">dtRow</param>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
		}



		#endregion
	}
}